<?php

namespace App\Http\Controllers\Payment;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\CustomerPackage;
use App\Models\SellerPackage;
use App\Models\CombinedOrder;
use App\Http\Controllers\CustomerPackageController;
use App\Http\Controllers\SellerPackageController;
use App\Http\Controllers\WalletController;
use App\Http\Controllers\CheckoutController;
use App\Models\Order;
use Session;
use Auth;

class RupantorPayController extends Controller
{
    protected $api_key;
    protected $api_url = 'https://payment.rupantorpay.com/api/payment/checkout';
    protected $verify_url = 'https://payment.rupantorpay.com/api/payment/verify-payment';

    public function __construct()
    {
        $this->api_key = env('RUPANTORPAY_API_KEY'); // Set this in your .env
    }

    public function pay()
    {
        if (Auth::user()->phone == null) {
            flash('Please add phone number to your profile')->warning();
            return redirect()->route('profile');
        }

        $email = Auth::user()->email ?? 'customer@example.com';

        $amount = 0;
        if (Session::has('payment_type')) {
            $paymentType = Session::get('payment_type');
            $paymentData = Session::get('payment_data');

            if ($paymentType == 'cart_payment') {
                $combined_order = CombinedOrder::findOrFail(Session::get('combined_order_id'));
                $amount = round($combined_order->grand_total, 2);
            } elseif ($paymentType == 'order_re_payment') {
                $order = Order::findOrFail($paymentData['order_id']);
                $amount = round($order->grand_total, 2);
            } elseif ($paymentType == 'wallet_payment') {
                $amount = round($paymentData['amount'], 2);
            } elseif ($paymentType == 'customer_package_payment') {
                $customer_package = CustomerPackage::findOrFail($paymentData['customer_package_id']);
                $amount = round($customer_package->amount, 2);
            } elseif ($paymentType == 'seller_package_payment') {
                $seller_package = SellerPackage::findOrFail($paymentData['seller_package_id']);
                $amount = round($seller_package->amount, 2);
            }
        } else {
            flash('Payment type is not set')->error();
            return redirect()->route('cart');
        }

        $postData = [
            'fullname'    => Auth::user()->name,
            'email'       => $email,
            'amount'      => (string) $amount, // RupantorPay expects string amount without trailing zeros for whole numbers
            'success_url' => route('rupantorpay.success'),
            'cancel_url'  => route('rupantorpay.cancel'),
            'webhook_url' => route('rupantorpay.webhook'), // optional, implement webhook route if needed
            'meta_data'   => json_encode([
                'payment_type' => $paymentType,
                'combined_order_id' => Session::get('combined_order_id'),
                'payment_data' => Session::get('payment_data'),
                'user_id' => Auth::id(),
            ]),
        ];

        $response = $this->createPaymentUrl($postData);

        if ($response && isset($response['status']) && $response['status'] == 1 && isset($response['payment_url'])) {
            return redirect()->away($response['payment_url']);
        } else {
            flash('Failed to create payment link. Please try again.')->error();
            return redirect()->route('cart');
        }
    }

    protected function createPaymentUrl(array $data)
    {
        $ch = curl_init();

        curl_setopt_array($ch, [
            CURLOPT_URL => $this->api_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'X-API-KEY: ' . $this->api_key,
                'X-CLIENT: ' . request()->getHost(),
            ],
            CURLOPT_SSL_VERIFYPEER => false,
        ]);

        $result = curl_exec($ch);

        if (curl_errno($ch)) {
            curl_close($ch);
            return null;
        }

        curl_close($ch);

        return json_decode($result, true);
    }

    public function success(Request $request)
    {
        // Retrieve transactionId from GET parameters
        $transactionId = $request->query('transactionId');

        if (!$transactionId) {
            flash('Invalid payment confirmation.')->error();
            return redirect()->route('cart');
        }

        $verification = $this->verifyPayment($transactionId);

        if ($verification && isset($verification['status']) && $verification['status'] === 'COMPLETED') {
            $paymentType = $verification['metadata']['payment_type'] ?? null;
            $combinedOrderId = $verification['metadata']['combined_order_id'] ?? null;
            $paymentData = $verification['metadata']['payment_data'] ?? null;

            if ($paymentType == 'cart_payment' && $combinedOrderId) {
                return (new CheckoutController)->checkout_done($combinedOrderId, json_encode($verification));
            } elseif ($paymentType == 'order_re_payment') {
                return (new CheckoutController)->orderRePaymentDone(json_decode($paymentData, true), json_encode($verification));
            } elseif ($paymentType == 'wallet_payment') {
                return (new WalletController)->wallet_payment_done(json_decode($paymentData, true), json_encode($verification));
            } elseif ($paymentType == 'customer_package_payment') {
                return (new CustomerPackageController)->purchase_payment_done(json_decode($paymentData, true), json_encode($verification));
            } elseif ($paymentType == 'seller_package_payment') {
                return (new SellerPackageController)->purchase_payment_done(json_decode($paymentData, true), json_encode($verification));
            } else {
                flash('Unknown payment type or missing data.')->error();
                return redirect()->route('cart');
            }
        } else {
            flash('Payment verification failed or payment not completed.')->error();
            return redirect()->route('cart');
        }
    }

    public function cancel()
    {
        flash('Payment cancelled.')->warning();
        return redirect()->route('cart');
    }

    // Optional webhook handler if you want server-to-server notification
    public function webhook(Request $request)
    {
        // Implement webhook handling logic if RupantorPay sends server notifications
        // Verify signature, check payment status, update your order/payment records accordingly
        return response()->json(['status' => 'success']);
    }

    protected function verifyPayment($transactionId)
    {
        $ch = curl_init();

        $postData = json_encode(['transaction_id' => $transactionId]);

        curl_setopt_array($ch, [
            CURLOPT_URL => $this->verify_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => $postData,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'X-API-KEY: ' . $this->api_key,
            ],
            CURLOPT_SSL_VERIFYPEER => false,
        ]);

        $result = curl_exec($ch);

        if (curl_errno($ch)) {
            curl_close($ch);
            return null;
        }

        curl_close($ch);

        return json_decode($result, true);
    }
}
